
// ====== Persistent Call Control Panel (Content Script) ======
const CCP_NS = "__ccp_panel__";
const Z = 2147483647;

// Style injector (put CSS in <head>)
const CCP_STYLE_ID = "__ccp_panel_style__";
function injectPanelStyles(){
  if (document.getElementById(CCP_STYLE_ID)) return;
  const css = `#${CCP_NS}{position:fixed;top:16px;right:16px;width:240px;max-width:92vw;min-width:180px;font-family:system-ui,-apple-system,Segoe UI,Roboto,Arial,sans-serif;z-index:${Z};color:#111;background:#ffffffee;backdrop-filter:blur(6px);border:1px solid #e5e7eb;border-radius:12px;box-shadow:0 10px 30px rgba(0,0,0,.18)}
#${CCP_NS} *,#${CCP_NS} *::before,#${CCP_NS} *::after{box-sizing:border-box}
#${CCP_NS} .hdr{display:flex;align-items:center;gap:6px;padding:6px 8px;cursor:move;background:#111827;color:#fff;border-radius:12px 12px 0 0}
#${CCP_NS} .hdr .title{font-weight:700;font-size:12px;flex:1}
#${CCP_NS} .hdr .btn{cursor:pointer;padding:3px 6px;border-radius:6px;background:#374151;font-size:11px}
#${CCP_NS} .hdr .btn:hover{background:#4b5563}
#${CCP_NS} .body{padding:6px;display:flex;flex-direction:column;gap:6px}
#${CCP_NS} .row{display:flex;gap:6px}
#${CCP_NS} input[type="text"]{flex:1;padding:5px 7px;border:1px solid #d1d5db;border-radius:8px;font-size:12px}
#${CCP_NS} .dev{border:1px dashed #e5e7eb;padding:6px;border-radius:8px}
#${CCP_NS} .devhdr{display:flex;gap:6px;align-items:center}
#${CCP_NS} .devname{font-weight:600;font-size:12px;flex:1}
#${CCP_NS} .badge{font-size:10px;padding:2px 6px;border-radius:999px;background:#eef2ff;color:#3730a3}
#${CCP_NS} button{padding:5px 6px;background:#4f46e5;color:#fff;border-radius:6px;font-weight:600;cursor:pointer;font-size:11px;border:none}
#${CCP_NS} button.ghost{background:#e5e7eb;color:#111}
#${CCP_NS} button.warn{background:#dc2626}
#${CCP_NS} .btn-call{font-size:12px;padding:6px 8px;height:32px;min-width:96px}
#${CCP_NS} .foot{display:flex;align-items:center;justify-content:space-between;padding:6px 8px 8px}
#${CCP_NS} .resizer{position:absolute;right:0;bottom:0;width:12px;height:12px;cursor:nwse-resize}
#${CCP_NS}.collapsed .body,#${CCP_NS}.collapsed .foot{display:none}
#${CCP_NS}.collapsed{width:160px}
#${CCP_NS} .small{opacity:.85;font-size:11px}`;
  const st = document.createElement("style");
  st.id = CCP_STYLE_ID;
  st.type = "text/css";
  st.textContent = css;
  (document.head || document.documentElement).appendChild(st);
}

const STORE = {
  PANEL_ENABLED: "panel_enabled",
  PANEL_STATE: "panel_state", // {x,y,width,collapsed}
};

function createPanel(){
  if (document.getElementById(CCP_NS)) return document.getElementById(CCP_NS);
  injectPanelStyles();

  const root = document.createElement("div");
  root.id = CCP_NS;
  root.innerHTML = `
    <div class="hdr" id="dragHdr">
      <div class="title">Bảng điều khiển cuộc gọi</div>
      <div class="btn" id="btnCompact" title="Thu gọn/Mở rộng">–</div>
      <div class="btn" id="btnPin" title="Ghim (luôn hiển thị)">Ghim</div>
      <div class="btn" id="btnClose" title="Tắt">✕</div>
    </div>
    <div class="body">
      <div class="row">
        <input id="ccp_number" type="text" placeholder="Nhập hoặc dán SĐT…"/>
        <button class="ghost" id="btnPaste">Dán</button>
      </div>
      <div id="ccp_devices"></div>
    </div>
    <div class="foot small">
      <div id="ccp_hint">Đã đăng nhập: <span id="ccp_user">...</span></div>
      <div>
        <button class="ghost" id="btnLeft">Trái</button>
        <button class="ghost" id="btnRight">Phải</button>
      </div>
    </div>
    <div class="resizer" id="resizer"></div>
  `;

  document.documentElement.appendChild(root);
  wirePanel(root);
  refreshDevices();
  return root;
}

let dragState = null;
function wirePanel(root){
  const hdr = root.querySelector("#dragHdr");
  hdr.addEventListener("mousedown", (e) => {
    dragState = {dx: e.clientX - root.offsetLeft, dy: e.clientY - root.offsetTop};
    e.preventDefault();
  });
  window.addEventListener("mousemove", (e) => {
    if (!dragState) return;
    root.style.left = (e.clientX - dragState.dx) + "px";
    root.style.top  = (e.clientY - dragState.dy) + "px";
    root.style.right = "auto";
  });
  window.addEventListener("mouseup", () => {
    if (dragState){ savePanelState(); dragState = null; }
  });

  root.querySelector("#btnClose").addEventListener("click", async () => {
    await chrome.storage.local.set({[STORE.PANEL_ENABLED]: false});
    root.remove();
  });
  root.querySelector("#btnCompact").addEventListener("click", async () => {
    root.classList.toggle("collapsed");
    await savePanelState();
  });
  root.querySelector("#btnLeft").addEventListener("click", async () => {
    root.style.left = "16px"; root.style.right="auto"; await savePanelState();
  });
  root.querySelector("#btnPin").addEventListener("click", async () => {
    await chrome.storage.local.set({[STORE.PANEL_ENABLED]: true});
  });
  root.querySelector("#btnRight").addEventListener("click", async () => {
    root.style.right = "16px"; root.style.left="auto"; await savePanelState();
  });
  root.querySelector("#btnPaste").addEventListener("click", async () => {
    try{
      const t = await navigator.clipboard.readText();
      const inp = document.getElementById("ccp_number");
      if (inp) inp.value = (t||"").trim();
    }catch(e){ /* ignore */ }
  });
  // Resizer (width only)
  const res = root.querySelector("#resizer");
  let resizing = false, startX = 0, startW = 0;
  res.addEventListener("mousedown", (e) => {
    resizing = true; startX = e.clientX; startW = root.offsetWidth; e.preventDefault();
  });
  window.addEventListener("mousemove", (e) => {
    if (!resizing) return;
    const dw = e.clientX - startX;
    const w = Math.min(Math.max(startW + dw, 180), Math.min(window.innerWidth - 40, 520));
    root.style.width = w + "px";
  });
  window.addEventListener("mouseup", async () => {
    if (resizing){ resizing = false; await savePanelState(); }
  });
}

async function savePanelState(){
  const el = document.getElementById(CCP_NS); if (!el) return;
  const rect = el.getBoundingClientRect();
  const state = {
    x: rect.left + window.scrollX,
    y: rect.top + window.scrollY,
    width: el.offsetWidth,
    collapsed: el.classList.contains("collapsed")
  };
  await chrome.storage.local.set({[STORE.PANEL_STATE]: state});
}

async function restorePanelState(){
  const el = document.getElementById(CCP_NS); if (!el) return;
  const st = await chrome.storage.local.get({[STORE.PANEL_STATE]: null});
  const s = st[STORE.PANEL_STATE];
  if (s){
    el.style.left = s.x + "px"; el.style.top = s.y + "px";
    el.style.right = "auto"; el.style.width = (s.width||240)+"px";
    if (s.collapsed) el.classList.add("collapsed"); else el.classList.remove("collapsed");
  }
}

// Build device list UI (single big button: "Dán & Gọi")
async function refreshDevices(){
  const wrap = document.getElementById("ccp_devices");
  if (!wrap) return;
  const cfg = await chrome.runtime.sendMessage({type:"ccp_get_cfg"}).catch(()=>null);
  const user = cfg && cfg.ok ? (cfg.data.auth_user || "") : "";
  const devices = cfg && cfg.ok ? (cfg.data.devices || []) : [];

  const userSpan = document.getElementById("ccp_user");
  if (userSpan) userSpan.textContent = user || "(chưa đăng nhập)";

  if (!devices.length){
    wrap.innerHTML = `<div class="small">Chưa có thiết bị. Vào <b>Tuỳ chọn</b> để thêm <i>Device Token</i> & tên hiển thị.</div>`;
    return;
  }

  const numberInput = document.getElementById("ccp_number");
  wrap.innerHTML = "";
  devices.forEach((d, idx) => {
    const dev = document.createElement("div");
    dev.className = "dev";
    dev.innerHTML = `
      <div class="devhdr">
        <div class="devname">${d.name || ("Thiết bị " + (idx+1))}</div>
        <div class="badge" id="st_${d.device_token}">IDLE</div>
      </div>
      <div class="row" style="margin-top:6px; gap:6px;">
        <button id="b_callsim1_${idx}" class="btn-call">Gọi SIM 1</button>
        <button id="b_callsim2_${idx}" class="btn-call ghost">Gọi SIM 2</button>
      </div>
    `;
    wrap.appendChild(dev);

    const btnSim1 = dev.querySelector(`#b_callsim1_${idx}`);
    const btnSim2 = dev.querySelector(`#b_callsim2_${idx}`);

    async function pasteAndCall(simSlot){
      let num = "";
      try { num = (await navigator.clipboard.readText()).trim(); } catch(_){}
      if (!num && numberInput) num = numberInput.value.trim();
      if (!num){ alert("Chưa có số để gọi."); return; }
      if (numberInput) numberInput.value = num;
      try{ await chrome.runtime.sendMessage({type:"ccp_paste", payload: {device_token:d.device_token, number:num}}); }catch(_){}
      const res = await chrome.runtime.sendMessage({type:"ccp_call", payload: {device_token:d.device_token, number:num, sim: simSlot}});
      if (res && res.ok && numberInput) { numberInput.value = ""; }
      if (!res || !res.ok) console.warn("Call error:", res && res.error);
    }

    if (btnSim1){ btnSim1.addEventListener("click", () => pasteAndCall(1)); }
    if (btnSim2){ btnSim2.addEventListener("click", () => pasteAndCall(2)); }

  });
}

// Init or toggle
async function ensurePanel(){
  const st = await chrome.storage.local.get({[STORE.PANEL_ENABLED]: false});
  if (!st[STORE.PANEL_ENABLED]) return;
  createPanel();
  await restorePanelState();
}
ensurePanel();

// Listen for toggle/show/refresh
chrome.runtime.onMessage.addListener((msg) => {
  if (!msg) return;
  if (msg.type === "ccp_toggle_panel"){
    chrome.storage.local.get({[STORE.PANEL_ENABLED]: false}).then(v => {
      const enabled = !v[STORE.PANEL_ENABLED];
      chrome.storage.local.set({[STORE.PANEL_ENABLED]: enabled}).then(() => {
        if (enabled) ensurePanel(); else { const el = document.getElementById(CCP_NS); if (el) el.remove(); }
      });
    });
  }
  if (msg.type === "ccp_show_panel"){
    chrome.storage.local.set({[STORE.PANEL_ENABLED]: true}).then(()=> ensurePanel());
  }
  if (msg.type === "ccp_refresh_devices"){
    refreshDevices();
  }
});

// ===== SSE subscribe (optional; update device status) =====
let __sse = null;
async function startSSE(){
  try{
    const cfgRes = await chrome.runtime.sendMessage({type:"ccp_get_cfg"});
    if (!cfgRes || !cfgRes.ok) return;
    const cfg = cfgRes.data;
    const user = cfg.auth_user || "";
    const token = (cfg.x_token||"").trim();
    const base = (cfg.base_origin || "") || "http://45.252.248.148:8010";
    if (!user || !token) return;

    const origin = (() => { try { return new URL(base).origin; } catch(_){ return base.replace(/\/+$/,""); } })();
    const url = `${origin}/events/stream?user=${encodeURIComponent(user)}&token=${encodeURIComponent(token)}&ts=${Date.now()}`;
    if (__sse) { try{ __sse.close(); }catch(_){ } __sse = null; }
    __sse = new EventSource(url);

    __sse.onmessage = (ev) => {
      try{
        const data = JSON.parse(ev.data);
        if (!data) return;
        const t = String(data.type||"");
        const devTok = data.device_token || (data.call && data.call.device_token);
        if (!devTok) return;
        const el = document.getElementById(`st_${devTok}`);
        if (!el) return;
        if (/start|ring|calling|call_new/i.test(t)) { el.textContent = "ĐANG GỌI"; el.style.background="#fee2e2"; el.style.color="#991b1b"; }
        else if (/end|hang|finish/i.test(t)) { el.textContent = "KẾT THÚC"; el.style.background="#ecfccb"; el.style.color="#365314"; setTimeout(()=>{el.textContent="IDLE"; el.style=null;}, 6000); }
        else { el.textContent = t.toUpperCase(); }
      }catch(_){}
    };
    __sse.onerror = (_)=>{
      try{ __sse && __sse.close(); }catch(_){}
      __sse = null;
      setTimeout(startSSE, 1800 + Math.random()*1200);
    };
  }catch(_){ /* ignore */ }
}
startSSE();
